/**
 * @package:   Part of vpl-jail-system
 * @copyright: Copyright (C) 2021 Juan Carlos Rodríguez-del-Pino
 * @license:   GNU/GPL, see LICENSE.txt or http://www.gnu.org/licenses/gpl-3.0.html
 **/

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <climits>
#include <limits>
#include <cstring>
#include <algorithm>
#include <fstream>
#include <sstream>
#include <exception>
#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/prctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pty.h>
#include <termios.h>
#include <sched.h>
#include <sys/mount.h>
#include <sys/wait.h>
#include <sys/syscall.h>
#include "jail.h"

// pivot_root syscall wrapper (not in glibc)
static int pivot_root(const char *new_root, const char *put_old) {
	return syscall(SYS_pivot_root, new_root, put_old);
}

namespace {
	static bool isOctalDigit(char c) {
		return c >= '0' && c <= '7';
	}

	// mountinfo escapes spaces, tabs, newlines as octal (e.g. "\\040")
	static std::string unescapeMountInfoPath(const std::string &in) {
		std::string out;
		out.reserve(in.size());
		for (size_t i = 0; i < in.size(); ++i) {
			if (in[i] == '\\' && i + 3 < in.size() && isOctalDigit(in[i + 1]) && isOctalDigit(in[i + 2]) && isOctalDigit(in[i + 3])) {
				int v = (in[i + 1] - '0') * 64 + (in[i + 2] - '0') * 8 + (in[i + 3] - '0');
				out.push_back(static_cast<char>(v));
				i += 3;
				continue;
			}
			out.push_back(in[i]);
		}
		return out;
	}

	static std::vector<std::string> listMountPoints() {
		std::vector<std::string> mountPoints;
		std::ifstream in("/proc/self/mountinfo");
		std::string line;
		while (std::getline(in, line)) {
			// Fields: id parent major:minor root mount_point ...
			std::istringstream iss(line);
			std::string id, parent, majmin, root, mountPoint;
			if (!(iss >> id >> parent >> majmin >> root >> mountPoint)) {
				continue;
			}
			mountPoints.push_back(unescapeMountInfoPath(mountPoint));
		}
		return mountPoints;
	}

	static bool isPathUnderPrefix(const std::string &path, const std::string &prefix) {
		if (path == prefix) {
			return true;
		}
		if (path.size() <= prefix.size()) {
			return false;
		}
		return (path.compare(0, prefix.size(), prefix) == 0 && path[prefix.size()] == '/');
	}

	static void unmountMountTreeOrThrow(const std::string &prefixMountPoint) {
		std::vector<std::string> mountPoints = listMountPoints();
		std::vector<std::string> targets;
		targets.reserve(mountPoints.size());
		for (const std::string &mp : mountPoints) {
			if (isPathUnderPrefix(mp, prefixMountPoint)) {
				targets.push_back(mp);
			}
		}
		// Unmount deepest paths first so parent mount is not busy due to submounts.
		std::sort(targets.begin(), targets.end(), [](const std::string &a, const std::string &b) {
			return a.size() > b.size();
		});

		for (const std::string &mp : targets) {
			// We'll also hit prefixMountPoint itself in this loop due to sorting; that's fine.
			if (umount2(mp.c_str(), 0) != 0) {
				// If the mount disappeared meanwhile, treat as success.
				if (errno == EINVAL || errno == ENOENT) {
					continue;
				}
				Logger::log(LOG_ERR,
						"SECURITY: Failed to unmount mount under old root '%s': %s (errno=%d)",
						mp.c_str(), strerror(errno), errno);
				throw HttpException(internalServerErrorCode,
						"Failed to unmount old root mounts - security breach prevented");
			}
		}
	}
}
#include "redirector.h"
#include "httpServer.h"
#include "util.h"
#include "processMonitor.h"
#include "websocket.h"
#include "cgroup.h"
/*
 * @return string "ready", "busy", "offline"
 */
string Jail::commandAvailable(long long memRequested){
	Logger::log(LOG_INFO,"Memory requested %lld", memRequested);
	if (memRequested <= 0) {
		return "ready";
	}
	if(Util::fileExists("/etc/nologin")){ //System going shutdown
		return "offline";
	}else {
		char *mem = (char*)malloc(memRequested);
		if(mem == NULL) return "busy";
		else free(mem);
	}
	return "ready";
}

void Jail::checkFilesNameCorrectness(mapstruct files) {
	for(mapstruct::iterator i = files.begin(); i != files.end(); i++) {
		if (! Util::correctPath(i->first)) {
			Logger::log(LOG_ERR,"Incorrect filename '%s'", i->first.c_str());
			throw HttpException(badRequestCode, "Incorrect file name");
		}
	}
}

void Jail::saveParseFiles(processMonitor &pm, RPC &rpc) {
	pid_t pid=fork();
	if(pid==0){ //new process
		try {
			pm.becomePrisoner();
			mapstruct files = rpc.getFiles();
			mapstruct fileencoding = rpc.getFileEncoding();
			//Save files to execution dir, decode data if needed
			for(mapstruct::iterator i = files.begin(); i != files.end(); i++){
				string name = i->first;
				string data = i->second->getString();
				if ( fileencoding.find(name) != fileencoding.end()
						&& fileencoding[name]->getInt() == 1 ) {
					Logger::log(LOG_INFO, "Decoding file %s from b64", name.c_str());
					data = Base64::decode(data);
					if ( name.length() > 4 && name.substr(name.length() - 4, 4) == ".b64") {
						name = name.substr(0, name.length() - 4);
					}
				}
				Logger::log(LOG_INFO, "Write file %s data size %lu", name.c_str(), (long unsigned int)data.size());
				pm.writeFile(name, data);
			}
		}
		catch(...) {
			_exit(EXIT_FAILURE);
		}
		_exit(EXIT_SUCCESS);
	} else {
		int status;
		if (waitpid(pid, &status, 0) < 0) {
			throw HttpException(internalServerErrorCode, "Error saving files");
		}
		if (!WIFEXITED(status) || WEXITSTATUS(status) != EXIT_SUCCESS) {
			throw HttpException(internalServerErrorCode, "Error saving files");
		}
	}
}

void Jail::deleteFilesMarkedForDeletion(processMonitor &pm, RPC &rpc) {
	pid_t pid=fork();
	if(pid==0) { //new process
		pm.becomePrisoner();
		mapstruct filestodelete = rpc.getFileToDelete();
		mapstruct files = rpc.getFiles();
		for (mapstruct::iterator i = filestodelete.begin(); i != filestodelete.end(); i++) {
			string name = i->first;
			if (files.find(name) == files.end()) {
				Logger::log(LOG_INFO, "File '%s' not in upload list then not deleted", name.c_str());
				continue; // File not in the upload list so skip
			}
			Logger::log(LOG_INFO, "Delete file %s", name.c_str());
			pm.deleteFile(name);
		}
		_exit(EXIT_SUCCESS);
	} else {
		int status;
		if (waitpid(pid, &status, 0) < 0) {
			throw HttpException(internalServerErrorCode, "Error deleting files");
		}
		if (!WIFEXITED(status) || WEXITSTATUS(status) != EXIT_SUCCESS) {
			throw HttpException(internalServerErrorCode, "Error deleting files");
		}
	}
}

ExecutionLimits Jail::getParseExecutionLimits(RPC &rpc) {
	mapstruct parsedata = rpc.getData();
	ExecutionLimits executionLimits = Configuration::getConfiguration()->getLimits();
	Logger::log(LOG_INFO,"Reading parms");
	executionLimits.log("Config");
	const TreeNode* maxtime = parsedata["maxtime"];
	const TreeNode* maxfilesize = parsedata["maxfilesize"];
	const TreeNode* maxmemory = parsedata["maxmemory"];
	const TreeNode* maxprocesses = parsedata["maxprocesses"];
	if (maxtime != NULL) {
		executionLimits.maxtime = min(maxtime->getInt(), executionLimits.maxtime);
	}
	if (maxfilesize != NULL) {
		executionLimits.maxfilesize = min(Util::fixMemSize(maxfilesize->getLong()),
                                      	executionLimits.maxfilesize);
	}
	if (maxmemory != NULL) {
		executionLimits.maxmemory = min(Util::fixMemSize(maxmemory->getLong()),
								    	executionLimits.maxmemory);
	}
	if (maxprocesses != NULL) {
		executionLimits.maxprocesses = min(maxprocesses->getInt(), executionLimits.maxprocesses);
	}
	executionLimits.log("Request");
	return executionLimits;
}

void Jail::commandRequest(RPC &rpc, string &adminticket,string &monitorticket,string &executionticket){
	mapstruct parsedata = rpc.getData();
	Logger::log(LOG_INFO,"Request for process");
	checkFilesNameCorrectness(rpc.getFiles());
	checkFilesNameCorrectness(rpc.getFileToDelete());
	processMonitor pm(adminticket, monitorticket, executionticket);
	// TODO: Cheking file names before the fork will give a proper response to the caller or not?
	pid_t pid=fork();
	if(pid==0){ //new process
		try {
			Logger::log(LOG_INFO,"Parse data %lu", (long unsigned int)parsedata.size());
			saveParseFiles(pm, rpc);
			Logger::log(LOG_INFO,"Reading parms");
			ExecutionLimits executionLimits = configuration->getLimits();
			string vpl_lang = parsedata["lang"]->getString();
			Logger::log(LOG_DEBUG, "VPL_LANG %s", vpl_lang.c_str());
			bool interactive = parsedata["interactive"]->getInt() > 0;
			Logger::log(LOG_DEBUG, "interactive %d", parsedata["interactive"]->getInt());
			pm.setExtraInfo(executionLimits, interactive, vpl_lang);
			pm.setCompiler();
			Logger::log(LOG_INFO, "Compilation");
			string script = parsedata["execute"]->getString();
			string compilationOutput = run(pm, script);
			pm.setCompilationOutput(compilationOutput);
			bool compiled = pm.FileExists(VPL_EXECUTION) || pm.FileExists(VPL_WEXECUTION) || pm.FileExists(VPL_WEBEXECUTION);
			if (compiled) {
				deleteFilesMarkedForDeletion(pm, rpc);
				if (!interactive && pm.FileExists(VPL_EXECUTION)) {
					pm.setRunner();
					Logger::log(LOG_INFO, "Non interactive execution");
					string program;
					if (pm.installScript(".vpl_launcher.sh", "vpl_batch_launcher.sh")) {
						program = ".vpl_launcher.sh";
					} else {
						program = VPL_EXECUTION;
					}
					string executionOutput = run(pm, program);
					Logger::log(LOG_INFO, "Write execution result");
					pm.setExecutionOutput(executionOutput, true);
				}
				executionLimits = getParseExecutionLimits(rpc);
				pm.setExtraInfo(executionLimits, interactive, vpl_lang);
			}else{
				Logger::log(LOG_INFO, "Compilation fail");
				for(int i = 0; !pm.isMonitored() && i < 30; i++){
					usleep(100000); //wait until monitor start and send compilation output
				}
			}
		}
		catch(std::exception &e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e.what(), __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(string &e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e.c_str(), __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(HttpException &e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e.getLog().c_str(), __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(const char *e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e,__FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(...){
			Logger::log(LOG_ERR, "unexpected exception %s:%d", __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		_exit(EXIT_SUCCESS);
	}
}

void Jail::commandDirectRun(RPC &rpc, string &homepath, string &adminticket, string &executionticket){
	mapstruct parsedata = rpc.getData();
	Logger::log(LOG_INFO,"Request for direct run");
	checkFilesNameCorrectness(rpc.getFiles());
	checkFilesNameCorrectness(rpc.getFileToDelete());
	processMonitor pm(adminticket, executionticket);
	homepath = pm.getRelativeHomePath();
	pid_t pid = fork();
	if (pid == 0) { //new process
		try {
			Logger::log(LOG_INFO,"Parse data %lu", (long unsigned int)parsedata.size());
			saveParseFiles(pm, rpc);
			Logger::log(LOG_INFO,"Reading parms");
			ExecutionLimits executionLimits = configuration->getLimits();
			string vpl_lang = parsedata["lang"]->getString();
			Logger::log(LOG_DEBUG, "VPL_LANG %s", vpl_lang.c_str());
			bool interactive = true;
			pm.setExtraInfo(executionLimits, interactive, vpl_lang);
			pm.setCompiler();
			Logger::log(LOG_INFO, "Compilation");
			string script = parsedata["execute"]->getString();
			string compilationOutput = run(pm, script);
			pm.setCompilationOutput(compilationOutput);
			bool compiled = pm.FileExists(VPL_EXECUTION);
			if (compiled) {
				deleteFilesMarkedForDeletion(pm, rpc);
				executionLimits = getParseExecutionLimits(rpc);
				pm.setExtraInfo(executionLimits, interactive, vpl_lang);
			}else{
				Logger::log(LOG_INFO, "Compilation fail");
			}
		}
		catch(std::exception &e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e.what(), __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(string &e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e.c_str(), __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(HttpException &e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e.getLog().c_str(), __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(const char *e){
			Logger::log(LOG_ERR, "unexpected exception: %s %s:%d", e,__FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		catch(...){
			Logger::log(LOG_ERR, "unexpected exception %s:%d", __FILE__, __LINE__);
			_exit(EXIT_FAILURE);
		}
		_exit(EXIT_SUCCESS);
	}
}


void Jail::commandGetResult(string adminticket, string &compilation,
                            string &execution, bool &executed,
							bool &interactive){
	processMonitor pm(adminticket);
	pm.getResult(compilation, execution, executed);
	interactive = pm.isInteractive();
}

bool Jail::commandUpdate(string adminticket, RPC &rpc){
	checkFilesNameCorrectness(rpc.getFiles());
	processMonitor pm(adminticket);
	try {
		saveParseFiles(pm, rpc);
		return true;
	} catch(...) {
		return false;
	}
}

bool Jail::commandRunning(string adminticket){
	try{
		processMonitor pm(adminticket);
		bool running = pm.isRunnig();
		if ( ! running ) {
			pm.cleanTask();
		}
		return running;
	}catch(...){
		return false;
	}
}
void Jail::commandStop(string adminticket){
	pid_t pid=fork();
	if(pid==0){ //new process
		try{
			processMonitor pm(adminticket);
			pm.cleanTask();
		}catch(...){

		}
		_exit(EXIT_SUCCESS);
	}
}
void Jail::commandMonitor(string monitorticket, Socket *s) {
	processMonitor pm(monitorticket);
	webSocket ws(s);
	processState state = prestarting;
	bool webserver = false;
	bool runBrowser = false;
	time_t startTime = 0;
	time_t lastMessageTime = 0;
	time_t lastTime = pm.getStartTime();
	string lastMessage;
	while (state != stopped) {
		processState newstate = pm.getState();
		time_t now = time(NULL);
		time_t timeout = pm.getStartTime() + pm.getMaxTime();
		if( ! lastMessage.empty() && now != lastMessageTime){
			ws.send(lastMessage + ": " + Util::itos(now-startTime) + " sec");
			lastMessageTime = now;
		}
		if (newstate != state) {
			state = newstate;
			switch(state) {
			case prestarting:
				break;
			case starting:
				Logger::log(LOG_DEBUG, "Monitor starting");
				startTime = now;
				lastMessageTime = now;
				lastMessage = "message:starting";
				ws.send(lastMessage);
				break;
			case compiling:
				Logger::log(LOG_DEBUG, "Monitor compiling");
				timeout = now + pm.getMaxTime();
				startTime = now;
				lastMessageTime = now;
				lastMessage = "message:compilation";
				ws.send(lastMessage);
				break;
			case beforeRunning:
				Logger::log(LOG_DEBUG, "Monitor beforeRunning");
				ws.send("compilation:" + pm.getCompilation());
				timeout = now + JAIL_SOCKET_TIMEOUT;
				if (pm.FileExists(VPL_EXECUTION)) {
					Logger::log(LOG_DEBUG, "run:terminal");
					ws.send("run:terminal");
				} else if (pm.FileExists(VPL_WEBEXECUTION)) {
					Logger::log(LOG_DEBUG, "run:webterminal");
					ws.send("run:webterminal");
					webserver = true;
				} else if (pm.FileExists(VPL_WEXECUTION)) {
					ws.send("run:vnc:" + pm.getVNCPassword());
				} else {
					Logger::log(LOG_DEBUG, "No executable to run");
					usleep(200000); //give time to save compilation output
					string compilationOutput = pm.getCompilation();
					if (compilationOutput.empty()) {
						ws.send("compilation:The compilation process did not generate an executable nor error message.");
					} else {
						ws.send("compilation:" + compilationOutput);
					}
					ws.send("close:");
					ws.close();
					ws.wait(500); //wait client response
					pm.cleanTask();
					ws.receive();
					return;
				}
				usleep(200000); //give time to save compilation output
				ws.send("compilation:" + pm.getCompilation());
				break;
			case running:
				Logger::log(LOG_DEBUG, "Monitor running");
				startTime = now;
				timeout = now + pm.getMaxTime() + 6 /* execution cleanup */;
				lastMessageTime = now;
				lastMessage = "message:running";
				ws.send(lastMessage);
				// TODO Checks if browser with cookie running to remove URL message
				if ( webserver && !runBrowser && pm.FileExists(VPL_LOCALSERVERADDRESSFILE) ) {
					runBrowser = true;
					ws.send("run:browser:" + pm.getHttpPassthroughTicket());
				}
				break;
			case retrieve:
				Logger::log(LOG_DEBUG, "Monitor retrieve");
				startTime = now;
				timeout = now + JAIL_HARVEST_TIMEOUT;
				ws.send("retrieve:");
				break;
			case stopped:
				Logger::log(LOG_DEBUG, "Monitor stopped");
				ws.send("close:");
				ws.close();
				ws.wait(500); //wait client response
				break;
			}
		}
		ws.wait(100); // 10 times a second
		
		string rec = ws.receive();
		if (ws.isClosed()) {
			if (state == retrieve && timeout >= time(NULL)) {
				continue;
			}
			break;
		}

		if (rec.size() > 0) { //Receive client close ws
			ws.close();
			break;
		}

		//Check running timeout
		if (state != starting && timeout < time(NULL)) {
			ws.send("message:timeout");
			Util::sleep(3000000);
			ws.send("close:");
			ws.close();
			ws.wait(500); //wait client response
			ws.receive();
			break;
		}

		if (lastTime != now && pm.isOutOfMemory()) { //Every second check memory usage
			string ml = pm.getMemoryLimit();
			Logger::log(LOG_DEBUG, "Out of memory (%s)", ml.c_str());
			ws.send("message:outofmemory:" + ml);
			Util::sleep(1500000);
			ws.send("close:");
			ws.close();
			ws.wait(500); //wait client response
			ws.receive();
			break;
		}
		lastTime = now;
	}
	pm.cleanTask();
}

void Jail::commandExecute(string executeticket, Socket *s){
	processMonitor pm(executeticket);
	webSocket ws(s);
	if (pm.getSecurityLevel() != execute){ 
		Logger::log(LOG_ERR,"%s: Security. Try to execute request with no monitor ticket",IP.c_str());
		throw "Internal server error";
	}
	Logger::log(LOG_INFO,"Start executing");
	if (pm.getState() == beforeRunning) { 
		pm.setRunner();
		if (pm.FileExists(VPL_EXECUTION)) {
			string program;
			if (pm.installScript(".vpl_launcher.sh", "vpl_terminal_launcher.sh")) {
				program = ".vpl_launcher.sh";
			} else {
				program = VPL_EXECUTION;
			}
			runTerminal(pm, ws, program);
		} else if (pm.FileExists(VPL_WEBEXECUTION)) {
			string program;
			if (pm.installScript(".vpl_launcher.sh", "vpl_web_launcher.sh")) {
				program = ".vpl_launcher.sh";
			} else {
				program = VPL_WEBEXECUTION;
			}
			runTerminal(pm, ws, program);
		} else if (pm.FileExists(VPL_WEXECUTION)) {
			if (pm.installScript(".vpl_launcher.sh", "vpl_vnc_launcher.sh"))
				runVNC(pm, ws, ".vpl_launcher.sh");
			else
				Logger::log(LOG_ERR, "%s:Error: vpl_vnc_launcher.sh not installed", IP.c_str());
		} else {
			Logger::log(LOG_ERR, "%s:Error: nothing to run", IP.c_str());
		}
	}
}

bool Jail::commandSetPassthroughCookie(string passthroughticket, HttpJailServer &server){
	try {
		processMonitor pm(passthroughticket);
		if ( pm.getState() != processState::running ||
		     pm.getSecurityLevel() != httppassthrough) {
			 return false;
		}
	} catch(...) {
		return false;
	}

	string extra = VPL_SETWEBCOOKIE + passthroughticket + "; Path=/\r\n";
	extra += VPL_LOCALREDIRECT;
	server.send(302, "REDIRECT", "", false, extra);
	return true;
}

/**
 * Passthrough the request to the localwebserver
 * 
 * @param socket The open socket that manage the connection
 * 
 * @return if the request finnaly has rigth to access
 */
bool Jail::httpPassthrough(string passthroughticket, Socket *socket){
	try {
		processMonitor pm(passthroughticket);
		if ( pm.getState() != processState::running ) {
			Logger::log(LOG_INFO,"httpPassthrough fail: ! processState::running");
			return false;
		}
		if ( pm.getSecurityLevel() != httppassthrough ) {
			Logger::log(LOG_INFO,"httpPassthrough fail: pm.getSecurityLevel() != httppassthrough");
			return false;
		}
		runPassthrough(pm, socket);
		return true;
	} catch(HttpException &exception){
		Logger::log(LOG_ERR,"%s:%s",IP.c_str(),exception.getLog().c_str());
		return false;
	}
	catch(std::exception &e){
		Logger::log(LOG_ERR,"%s:Unexpected exception %s on %s:%d",IP.c_str(), e.what(),__FILE__,__LINE__);
	}
	catch(string &exception){
		Logger::log(LOG_ERR,"%s:%s",IP.c_str(),exception.c_str());
	}
	catch(const char *s){
		Logger::log(LOG_ERR,"%s:%s",IP.c_str(),s);
	}catch(...) {
		Logger::log(LOG_INFO,"httpPassthrough fail: unexpected exception");
	}
	return false;
}

const vplregex Jail::regPassthroughTicket("^\\/([^\\/]+)\\/httpPassthrough$");
/**
 * Checks for http Passthrough setting request
 * 
 * @param URLPath URL path to check
 * @param ticket possible ticket found
 * 
 * @return true if valid request structure found
 */
bool Jail::isRequestingCookie(string URLPath, string &ticket) {
	vplregmatch match(3);
	if (! regPassthroughTicket.search(URLPath, match)) {
		return false;
	}
	ticket = match[1];
	return true;
}

/**
 * Returns if IP can request a task
 * @return true if IP is allow to request task
 */
bool Jail::isValidIPforRequest() {
	const vector<string> &dirs = configuration->getTaskOnlyFrom();
	if (dirs.size() == 0) return true;
	int l = (int) dirs.size();
	for(int i = 0; i< l; i++){
		if(IP.find(dirs[i]) == 0)
			return true;
	}
	return false;
}


/**
 * Constructor, parameters => main parameters
 */
Jail::Jail(string IP){
	this->IP = IP;
	this->newpid = -1;
	this->redirectorpid = -1;
	configuration = Configuration::getConfiguration();
}

const vplregex Jail::regChallenge("^\\/\\.well-known\\/acme-challenge\\/([^\\/]*)$");

string Jail::predefinedURLResponse(string URLPath) {
	string page;
	if( Util::toUppercase(URLPath) == "/OK"){
		page = "<!DOCTYPE html><html><body><h1>OK</h1></body></html>";
	} else if( URLPath == "/robots.txt"){
		page = "User-agent: *\nDisallow: /\n";
	} else if( URLPath == "/favicon.ico"){
		page = "<svg viewBox=\"0 0 25 14\" xmlns=\"http://www.w3.org/2000/svg\">"
				"<style> .logo { font: bold 12px Arial Rounded MT,Arial,Helvetica;"
				"fill: #277ab0; stroke: black; stroke-width: 0.7px;} </style>"
				"<text x=\"0\" y=\"12\" class=\"logo\">VPL</text></svg>";
	} else if(configuration->getCertbotWebrootPath() != "") {
		static vplregmatch path(2);
		if (regChallenge.search(URLPath, path)) {
			string filePath = configuration->getCertbotWebrootPath() + URLPath;
			if (Util::fileExists(filePath)) {
				page = Util::readFile(filePath, false, configuration->getCertbotWebrootPath().size() + 1);
			}
		}
	}
	return page;
}

const vplregex Jail::regWebSocketPath("^\\/([^\\/]+)\\/(.+)$");

/**
 * Process request
 *  3) read http request and header
 *  4) select http xmlrpc or websocket
 *  5) http: available, request, getresult, running, stop
 *  6) websocket: monitor, execute
 */
void Jail::process(Socket *socket){
	Logger::log(LOG_INFO, "Start server version %s", Util::version());
	string httpURLPath = configuration->getURLPath();
	HttpJailServer server(socket);
	try {
		socket->readHeaders();
		if(socket->headerSize() == 0){
			if(socket->isSecure()){ // Don't count SSL errors.
				_exit(static_cast<int>(neutral));
			}else{
				_exit(EXIT_FAILURE);
			}
		}
		if (socket->getCookie(VPL_WEBCOOKIE).length()) { // If found cookie passing to local application and stop processing
			if (httpPassthrough(socket->getCookie(VPL_WEBCOOKIE), socket)) {
				_exit(EXIT_SUCCESS);
			}
		}
		string httpMethod = socket->getMethod();
		if (Util::toUppercase(socket->getHeader("Upgrade")) != "WEBSOCKET") {
			Logger::log(LOG_INFO, "http(s) request");
			if (httpMethod == "GET") {
				ExitStatus securityStatus = ExitStatus::neutral;
				string response;
				string cookieTicket;
				if (isRequestingCookie(socket->getURLPath(), cookieTicket)) { // Getting cookie to access web app
					string iWasHere = socket->getCookie(VPL_IWASHERECOOKIE);
					if ( (!iWasHere.empty()) &&
					     (socket->getQueryString() == "private")) { // Requires a private browser
						response = "<html><body><h1>Please, to access web applications use a new Private or Incognito window.</h1></body></html>";
					} else {
						if (! commandSetPassthroughCookie(cookieTicket, server)) {
							securityStatus = ExitStatus::httpError;
						}
					}
				}
				if ( response.empty() ) {
					response = predefinedURLResponse(socket->getURLPath());
				}
				if ( response.empty() ) {
					throw HttpException(notFoundCode, "HTTP GET: URL path not found");
				}
				server.send200(response, false, VPL_SETIWASHERECOOKIE);
				_exit(static_cast<int>(securityStatus));
			}
			if (httpMethod == "HEAD") {
				string response = predefinedURLResponse(socket->getURLPath());
				if ( response.size() == 0 ) {
					throw HttpException(notFoundCode, "HTTP HEAD: URL path not found '" + socket->getURLPath() + "'");
				}
				server.send200(response, true);
				_exit(static_cast<int>(neutral));
			}
			if (httpMethod != "POST") {
				throw HttpException(badRequestCode, "Http(s) Unsupported METHOD " + httpMethod);
			}
			if (!isValidIPforRequest()) {
				throw HttpException(badRequestCode, "Client not allowed");
			}
			server.validateRequest(httpURLPath);
			string data = server.receive();
			RPC *prpc;
			if (data[0] == '<') {
				prpc = new XMLRPC(data);
			} else {
				prpc = new JSONRPC(data);
			}
			RPC& rpc = *prpc;
			string request = rpc.getMethodName();
			mapstruct parsedata = rpc.getData();
			Logger::log(LOG_INFO, "Execute request '%s'", request.c_str());
			if (request == "available") {
				ExecutionLimits jailLimits = configuration->getLimits();
				long long memRequested = parsedata["maxmemory"]->getLong();
				// Next line fixes XML-RPC int limits (-1).
				memRequested = memRequested > 0 ? memRequested : configuration->getLimits().maxmemory;
				string status = commandAvailable(memRequested);
				Logger::log(LOG_INFO, "Status: '%s'", status.c_str());
				server.send200(rpc.availableResponse(status,
				        processMonitor::requestsInProgress(),
						jailLimits.maxtime,
						jailLimits.maxfilesize,
						jailLimits.maxmemory,
						jailLimits.maxprocesses,
						configuration->getSecurePort()));
			} else if(request == "request") {
				Logger::log(LOG_NOTICE, "Executing Task from Server: '%s'", IP.c_str());
				string adminticket, monitorticket, executionticket;
				commandRequest(rpc, adminticket, monitorticket, executionticket);
				server.send200(rpc.requestResponse(adminticket,
								monitorticket,executionticket,
								configuration->getPort(),
								configuration->getSecurePort()));
			} else if(request == "directrun") {
				string homepath, adminticket, executionticket;
				commandDirectRun(rpc, homepath, adminticket, executionticket);
				server.send200(rpc.directRunResponse(homepath,
								adminticket,
								executionticket,
								configuration->getPort(),
								configuration->getSecurePort()));
			}else if(request == "getresult") {
				string adminticket, compilation, execution;
				bool executed,interactive;
				adminticket=parsedata["adminticket"]->getString();
				commandGetResult(adminticket, compilation, execution, executed, interactive);
				server.send200(rpc.getResultResponse(compilation, execution, executed, interactive));
			} else if(request == "running") {
				string adminticket;
				adminticket=parsedata["adminticket"]->getString();
				bool running = commandRunning(adminticket);
				if (! running && parsedata.count("pluginversion") == 0) {
					// Restores behaviour of < 2.3 version
					throw "Ticket invalid";
				}
				server.send200(rpc.runningResponse(running));
			} else if(request == "stop") {
				string adminticket;
				adminticket=parsedata["adminticket"]->getString();
				commandStop(adminticket);
				server.send200(rpc.stopResponse());
			} else if(request == "update") {
				string adminticket=parsedata["adminticket"]->getString();
				bool ok = commandUpdate(adminticket, rpc);
				server.send200(rpc.updateResponse(ok));
			} else { //Error
				throw HttpException(badRequestCode, "Unknown request:" + request);
			}
			delete prpc;
		} else { //Websocket
			if(socket->getMethod() != "GET"){
				throw "ws(s) Unsupported METHOD " + httpMethod;
			}
			string URLPath = socket->getURLPath();
			vplregmatch found(3);
			bool isfound = regWebSocketPath.search(URLPath, found);
			if (! isfound) {
				throw string("Bad URL");
			}
			string ticket = found[1];
			string command = found[2];
			if (command == "monitor") {
				commandMonitor(ticket, socket);
			} else if (command == "execute") {
				commandExecute(ticket, socket);
			} else {
				throw string("Bad command");
			}
		}
		_exit(EXIT_SUCCESS);
	}
	catch(HttpException &exception){
		Logger::log(LOG_INFO, "%s:%s", IP.c_str(), exception.getLog().c_str());
		server.sendCode(exception.getCode(), exception.getMessage());
		_exit(EXIT_SUCCESS);
	}
	catch(std::exception &e){
		Logger::log(LOG_WARNING, "%s:Unexpected exception %s on %s:%d", IP.c_str(), e.what(), __FILE__, __LINE__);
		server.sendCode(internalServerErrorCode, "Unknown error");
	}
	catch(string &exception){
		Logger::log(LOG_WARNING, "%s:%s",IP.c_str(), exception.c_str());
		server.sendCode(internalServerErrorCode, exception);
	}
	catch(const char *s){
		Logger::log(LOG_WARNING, "%s:%s",IP.c_str(), s);
		server.sendCode(internalServerErrorCode, s);
	}
	catch(...){
		Logger::log(LOG_WARNING, "%s:Unexpected exception %s:%d",IP.c_str(),__FILE__,__LINE__);
		server.sendCode(internalServerErrorCode,"Unknown error");
	}
	_exit(EXIT_FAILURE);
}

/**
 * Setup Linux namespaces for process isolation
 * Isolates PID, mounts and IPC namespaces
 */
void Jail::setupNamespaces() {
	// Check if namespace isolation is enabled
	if (!configuration->getUseNamespace()) {
		Logger::log(LOG_DEBUG, "Namespace isolation disabled in configuration");
		return;
	}
	
	// Create new namespaces to isolate the process
	// Using only PID and IPC namespaces
	// CLONE_NEWPID: PID namespace - isolated process IDs
	// CLONE_NEWNS: Mount namespace - isolated mount points
	// CLONE_NEWIPC: IPC namespace - isolated System V IPC, POSIX message queues
	int flags = CLONE_NEWPID | CLONE_NEWNS | CLONE_NEWIPC;
	if (unshare(flags) != 0) {
		// If unshare fails, log but continue (some kernels may not support all namespaces)
		Logger::log(LOG_WARNING, "Failed to create namespaces: %s (errno=%d). Continuing without full namespace isolation.", strerror(errno), errno);
		return;
	}
	// Prevent mount events from propagating back to the host (best-effort)
	if (mount(NULL, "/", NULL, MS_REC | MS_PRIVATE, NULL) != 0) {
		Logger::log(LOG_WARNING, "Failed to make mount tree private: %s (errno=%d)", strerror(errno), errno);
	}
	// Fork to apply the new PID namespace
	pid_t pid = fork();
	if (pid < 0) {
		Logger::log(LOG_ERR, "Failed to fork after unshare: %s (errno=%d)", strerror(errno), errno);
		throw HttpException(internalServerErrorCode, "I can't fork after unshare");
	}
	if (pid > 0) {
		// Parent process - exit when child finishes
		// Set dumpable to 0 for additional security (prevents ptrace attach)
		if (prctl(PR_SET_DUMPABLE, 0, 0, 0, 0) != 0) {
			Logger::log(LOG_WARNING, "Failed to set DUMPABLE to 0: %m");
		}
		// Prevent other processes from ptracing this one
		if (prctl(PR_SET_PTRACER, 0, 0, 0, 0) != 0) {
			Logger::log(LOG_WARNING, "Failed to set PTRACER to 0: %m");
		}
		// Wait for child to finish
		waitpid(pid, NULL, 0);
		_exit(EXIT_SUCCESS);
	}
	Logger::log(LOG_INFO, "Namespaces created: PID, NS, IPC, USER");
}

/**
 * Setup Linux USER namespace for process isolation
 * @param mp The process monitor
 */
void Jail::setupNamespaceUser(processMonitor &mp) {
	// Check if namespace isolation is enabled
	if (!configuration->getUseNamespace()) {
		Logger::log(LOG_DEBUG, "Namespace isolation disabled in configuration");
		return;
	}
	if (configuration->isRunningInContainer()) {
		Logger::log(LOG_DEBUG, "Running in container, skipping user namespace setup");
		return;
	}
	try{
		// Fix PTY ownership before unsharing so it is accessible inside the user namespace.
		// We set it to the prisoner UID so they own the terminal (stdin/stdout/stderr).
		uid_t prisonerUID = mp.getPrisonerID();
		if (isatty(STDIN_FILENO)) {
			if (fchmod(STDIN_FILENO, 0620) != 0) {
				Logger::log(LOG_WARNING, "Failed to chmod PTY slave: %m");
			}
			if (fchown(STDIN_FILENO, prisonerUID, -1) != 0) {
				Logger::log(LOG_WARNING, "Failed to chown PTY slave: %m");
			}
		}
	} catch(...) {
		// Ignore errors here, main logic is in the try block below
	}
	
	// CLONE_NEWUSER: User namespace - isolated user and group IDs
	int flags = CLONE_NEWUSER;
	if (unshare(flags) != 0) {
		// If unshare fails, log but continue (some kernels may not support all namespaces)
		Logger::log(LOG_WARNING, "Failed to create namespaces: %s (errno=%d). Continuing without full namespace isolation.", strerror(errno), errno);
		// Restore PTY ownership to root if we failed
		try{
			uid_t rootUGID = 0;
			if (isatty(STDIN_FILENO)) {
				if (fchown(STDIN_FILENO, rootUGID, -1) != 0) {
					Logger::log(LOG_WARNING, "Failed to chown PTY slave: %m");
				}
			}
		} catch(...) {
			// Ignore errors here
		}
		return;
	}
	try{
		uid_t fakeRootUGID = configuration->getHomeDirOwnerUid();
		// Write UID map to /proc/self/uid_map and /proc/self/setgroups, /proc/self/gid_map
		string uidMap = "0 " + Util::itos(fakeRootUGID) + " 1\n";
		uidMap += Util::itos(mp.getPrisonerID()) + " " + Util::itos(mp.getPrisonerID()) + " 1\n";
		Util::writeFile("/proc/self/uid_map", uidMap);
		Util::writeFile("/proc/self/gid_map", uidMap);
		Util::writeFile("/proc/self/setgroups", "deny"); // Prevent "setgroups" syscall in user namespace
	} catch(...) {
		Logger::log(LOG_WARNING, "Failed to set up UID/GID mappings for user namespace. Continuing without user namespace isolation.");
	}
	Logger::log(LOG_INFO, "Namespaces created: USER");
}

/**
 * Use pivot_root to change the root filesystem
 * This is more secure than chroot as it actually changes the root mount
 */
void Jail::pivotRoot(string jailPath, int prisonerID){
	if (jailPath == "") {
		Logger::log(LOG_INFO,"No pivot_root, running in container");
		return;
	}
	// Ensure mount propagation doesn't leak to the host (best-effort).
	// Doing it here too helps when setupNamespaces() is disabled.
	mount(NULL, "/", NULL, MS_REC | MS_PRIVATE, NULL);

	// Make the jail path a mount point by bind mounting it to itself.
	if (mount(jailPath.c_str(), jailPath.c_str(), NULL, MS_BIND | MS_REC, NULL) != 0) {
		Logger::log(LOG_WARNING, "Failed to bind mount jail path: %s. Falling back to chroot.", strerror(errno));
		// Fallback to chroot
		if(chdir(jailPath.c_str()) != 0)
			throw HttpException(internalServerErrorCode, "I can't chdir to jail", jailPath);
		if(chroot(jailPath.c_str()) != 0)
			throw HttpException(internalServerErrorCode, "I can't chroot to jail", jailPath);
		Logger::log(LOG_INFO,"chrooted (fallback) \"%s\"",jailPath.c_str());
		return;
	}
	// Make it private so mounts don't propagate (required for reliable cleanup).
	if (mount(NULL, jailPath.c_str(), NULL, MS_REC | MS_PRIVATE, NULL) != 0) {
		Logger::log(LOG_WARNING, "Failed to make jail path private: %s (errno=%d).", strerror(errno), errno);
	}
	
	// Create a unique directory for the old root using prisoner ID
	string oldroot = jailPath + "/.oldroot_p" + Util::itos(prisonerID);
	mkdir(oldroot.c_str(), 0755); // Ignore errors if it exists
	
	// Change to the new root
	if(chdir(jailPath.c_str()) != 0) {
		Logger::log(LOG_ERR, "Failed to chdir to jail: %s", strerror(errno));
		throw HttpException(internalServerErrorCode, "I can't chdir to jail", jailPath);
	}
	
	// Pivot the root
	string oldrootRelative = ".oldroot_p" + Util::itos(prisonerID);
	if (pivot_root(".", oldrootRelative.c_str()) != 0) {
		Logger::log(LOG_WARNING, "Failed to pivot_root: %s (errno=%d). Falling back to chroot.", strerror(errno), errno);
		// Fallback to chroot
		if(chroot(jailPath.c_str()) != 0)
			throw HttpException(internalServerErrorCode, "I can't chroot to jail", jailPath);
		Logger::log(LOG_INFO,"chrooted (fallback) \"%s\"",jailPath.c_str());
		return;
	}
	
	// Change to the new root
	if(chdir("/") != 0) {
		Logger::log(LOG_ERR, "Failed to chdir to new root: %s", strerror(errno));
		throw HttpException(internalServerErrorCode, "I can't chdir to new root");
	}
	
	// Unmount the old root immediately - CRITICAL for security
	// Use regular umount (not lazy MNT_DETACH) to fail fast if references exist
	string oldrootAbsolute = "/" + oldrootRelative;
	// EBUSY here is typically caused by nested mounts (e.g. /proc, /sys) living under the old root.
	// Unmount any submounts first, deepest-first, then unmount the old root mount point.
	unmountMountTreeOrThrow(oldrootAbsolute);
	
	// Remove the old root directory
	rmdir(oldrootAbsolute.c_str());
	Logger::log(LOG_INFO,"pivot_root completed to \"%s\"",jailPath.c_str());
}

/**
 * Remount /proc to show only prisoner processes within chroot
 * Note: Without PID namespace, /proc will show all processes, but chroot
 * provides some isolation. PIDs outside the prisoner's range are still visible
 * but their details may be restricted by file permissions.
 */
void Jail::remountProc(){
	string jailPath = configuration->getJailPath();
	
	// After chroot, paths are relative to new root
	const char* procPath = "/proc";
	
	// Unmount old /proc if it exists (ignore errors)
	umount2(procPath, MNT_DETACH);
	
	// Mount fresh /proc (combined with chroot for isolation)
	if (mount("proc", procPath, "proc", MS_NOSUID|MS_NOEXEC|MS_NODEV, NULL) != 0) {
		Logger::log(LOG_WARNING, "Failed to remount /proc: %s. Process visibility may not be isolated.", strerror(errno));
		return;
	}
	
	Logger::log(LOG_DEBUG, "Remounted /proc inside chroot");
	
	// Also remount /sys as read-only to limit information leakage
	const char* sysPath = "/sys";
	if (Util::fileExists(sysPath)) {
		umount2(sysPath, MNT_DETACH);
		if (mount("sysfs", sysPath, "sysfs", MS_RDONLY|MS_NOSUID|MS_NOEXEC|MS_NODEV, NULL) == 0) {
			Logger::log(LOG_DEBUG, "Remounted /sys as read-only");
		}
	}
}

/**
 * Make private per prisoner any writable dir in the jail to avoid information sharing.
 *
 * When mount namespaces are enabled, bind-mount /home/p<uid>/.writabledir onto /writabledir.
 */
void Jail::setupPrivateWritableDirs(processMonitor &pm){
    if (!configuration->getUseNamespace()) {
        return;
    }
    for (const string &dir : configuration->getWritableDirsInJail()) {
        // Source is /home/p<uid>/writable-dirname written with '.' instead of '/'
        string sanitizedDirName = dir;
        std::replace(sanitizedDirName.begin(), sanitizedDirName.end(), '/', '.');
        string source = pm.getRelativeHomePath() + "/" + sanitizedDirName;
        // Get original directory's permissions BEFORE creating source
        struct stat origStat;
        mode_t origMode = 0755; // Sensible default
        if (stat(dir.c_str(), &origStat) == 0) {
            origMode = origStat.st_mode & 07777;
            Logger::log(LOG_DEBUG, "Original %s has permissions %04o", dir.c_str(), origMode);
        } else {
            Logger::log(LOG_DEBUG, "Failed to stat original %s, using default 0755", dir.c_str());
        }
        // Create source with matching permissions
        if (mkdir(source.c_str(), origMode) != 0 && errno != EEXIST) {
            Logger::log(LOG_WARNING, "Failed to create private writable source %s: %s", source.c_str(), strerror(errno));
            continue;
        }       
        // Ensure permissions match (in case directory already existed)
        if (chmod(source.c_str(), origMode) != 0) {
            Logger::log(LOG_DEBUG, "Failed to chmod %s to %04o: %s", source.c_str(), origMode, strerror(errno));
        }
        if (chown(source.c_str(), pm.getPrisonerID(), pm.getPrisonerID()) != 0) {
            Logger::log(LOG_DEBUG, "Failed to chown %s: %s", source.c_str(), strerror(errno));
        }
        // Ensure mount point exists with original permissions
        if (mkdir(dir.c_str(), origMode) != 0 && errno != EEXIST) {
            Logger::log(LOG_WARNING, "Failed to ensure %s exists: %s", dir.c_str(), strerror(errno));
            continue;
        }
        
        // Bind mount private writable dir onto mount point
        if (mount(source.c_str(), dir.c_str(), NULL, MS_BIND, NULL) != 0) {
            Logger::log(LOG_WARNING, "Failed to bind-mount %s to %s: %s", source.c_str(), dir.c_str(), strerror(errno));
            continue;
        }
        
        // Make mount private (best-effort)
        mount(NULL, dir.c_str(), NULL, MS_PRIVATE, NULL);       
        // Best-effort hardening flags on the bind mount (ignore failures)
        mount(NULL, dir.c_str(), NULL, MS_BIND | MS_REMOUNT | MS_NODEV | MS_NOSUID, NULL);       
        Logger::log(LOG_DEBUG, "Private writable dir %s set to %s (mode %04o)", dir.c_str(), source.c_str(), origMode);
    }
}

/**
 * Setup cgroup isolation for resource control
 */
void Jail::setupCgroup(processMonitor &pm){
	if (!configuration->getUseCGroup()) {
		return;
	}
	
	try {
		string cgroupName = "p" + Util::itos(pm.getPrisonerID());
		Cgroup cgroup(cgroupName);
		cgroup.removeCgroup(); // Clean previous cgroups if exist
		// Add current process to cgroup controllers
		pid_t pid = getpid();
		// cgroup.setCPUProcs(pid);
		cgroup.setMemoryProcs(pid);
		// cgroup.setNetProcs(pid);
		
		// Set resource limits via cgroup
		ExecutionLimits limits = pm.getLimits();
		if (limits.maxmemory > 0) {
			cgroup.setMemoryLimitInBytes(limits.maxmemory);
			Logger::log(LOG_INFO, "Cgroup memory limit set to %lld bytes", limits.maxmemory);
		}
		
		Logger::log(LOG_INFO, "Process %d added to cgroup %s", pid, cgroupName.c_str());
	} catch (const std::exception &e) {
		Logger::log(LOG_WARNING, "Failed to setup cgroup: %s", e.what());
	} catch (const char *s) {
		Logger::log(LOG_WARNING, "Failed to setup cgroup: %s", s);
	} catch (const string &s) {
		Logger::log(LOG_WARNING, "Failed to setup cgroup: %s", s.c_str());
	} catch (...) {
		Logger::log(LOG_WARNING, "Failed to setup cgroup: unknown error");
	}
}

/**
 * chdir and chroot to jail with cgroup isolation
 */
void Jail::setupFilesystemIsolation(processMonitor &pm){
	string jailPath=configuration->getJailPath();
	
	if (jailPath == "") {
		Logger::log(LOG_INFO,"No pivot_root, running in container");
		return;
	}
	if (!configuration->getUseNamespace()) {
		// Without namespaces, fallback to chroot
		if(chdir(jailPath.c_str()) != 0)
			throw HttpException(internalServerErrorCode, "I can't chdir to jail", jailPath);
		if(chroot(jailPath.c_str()) != 0)
			throw HttpException(internalServerErrorCode, "I can't chroot to jail", jailPath);
		Logger::log(LOG_INFO,"chrooted \"%s\"",jailPath.c_str());
		return;
	}
	
	// Use pivot_root for stronger isolation than chroot
	pivotRoot(jailPath, pm.getPrisonerID());
	
	// After pivot_root, remount /proc to show only this namespace's processes
	remountProc();
	// Isolate writable dirs
	setupPrivateWritableDirs(pm);
}

/**
 * Execute program at prisoner home directory
 */
void Jail::transferExecution(processMonitor &pm, string fileName){
	string dir = pm.getRelativeHomePath();
	Logger::log(LOG_DEBUG, "Jail::transferExecution to %s+%s", dir.c_str(), fileName.c_str());
	string fullname = dir + "/" + fileName;
	if(chdir(dir.c_str())){
		throw "I can't chdir to exec dir :" + dir;
	}
	if(!Util::fileExists(fullname)){
		throw string("transferExecution: execution file not found: ") + fullname;
	}
	char *arg[6];
	char *command= new char[fullname.size() + 1];
	strcpy(command, fullname.c_str());
	const bool wantJobControl = (pm.isInteractive() && pm.getState() == processState::running);
	if (setsid() != 0) {
		// setsid() commonly fails with EPERM if we are already a process-group leader.
		Logger::log(LOG_DEBUG, "setsid() failed (non-fatal): %s (errno=%d)", strerror(errno), errno);
	}
	if (wantJobControl) {
		// Best-effort: (re)acquire controlling TTY on stdin.
		if (ioctl(STDIN_FILENO, TIOCSCTTY, 0) != 0) {
			Logger::log(LOG_DEBUG, "TIOCSCTTY failed (non-fatal): %s (errno=%d)", strerror(errno), errno);
		}
	}
	// Create a dedicated process group for the executed program (and its children).
	// This enables job control and group signaling.
	if (setpgid(0, 0) != 0) {
		Logger::log(LOG_DEBUG, "setpgid(0,0) failed (non-fatal): %s (errno=%d)", strerror(errno), errno);
	}
	if (wantJobControl && isatty(STDIN_FILENO)) {
		// Make this process group the foreground group of the controlling terminal.
		// If this fails, interactive programs may be stopped on read (SIGTTIN).
		if (tcsetpgrp(STDIN_FILENO, getpgrp()) != 0) {
			Logger::log(LOG_WARNING, "tcsetpgrp failed: %s (errno=%d)", strerror(errno), errno);
		}
	}
	int narg=0;
	arg[narg++] = command;
	arg[narg++] = NULL;
	int nenv = 0;
	char *env[12];
	string uid = Util::itos(pm.getPrisonerID());
	string HOME = "HOME=/home/p" + uid;
	env[nenv++] = (char *)HOME.c_str();
	string TMPDIR = "TMPDIR=/tmp";
	env[nenv++] = (char *)TMPDIR.c_str();
	string PATH = "PATH=" + configuration->getCleanPATH();
	env[nenv++] = (char *)PATH.c_str();
	env[nenv++] = (char *)"TERM=dumb";
	string UID = "UID=" + uid;
	env[nenv++]=(char *)UID.c_str();
	string USER = "USER=p" + uid;
	env[nenv++] = (char *)USER.c_str();
	string USERNAME = "USERNAME=Prisoner " + uid;
	env[nenv++] = (char *)USERNAME.c_str();
	string VPL_LANG = "VPL_LANG=" + pm.getLang();
	env[nenv++] = (char *)VPL_LANG.c_str();
	string VPL_VNCPASSWD = "VPL_VNCPASSWD=" + pm.getVNCPassword();
	if (pm.isInteractive()) {
		env[nenv++] = (char *)VPL_VNCPASSWD.c_str();
	}
	env[nenv++] = NULL;
	Logger::log(LOG_DEBUG, "Running \"%s\"", command);
	execve(command, arg, env);
	throw string("I can't execve: ") + command + " (" + strerror(errno) + ")";
}

/**
 * set user limits
 */
void Jail::setLimits(processMonitor &pm){
	ExecutionLimits executionLimits = pm.getLimits();
	executionLimits.log("setLimits");
	struct rlimit limit;
	limit.rlim_cur = 0;
	limit.rlim_max = 0;
	setrlimit(RLIMIT_CORE, &limit);
	limit.rlim_cur = executionLimits.maxtime;
	limit.rlim_max = executionLimits.maxtime;
	setrlimit(RLIMIT_CPU, &limit);
	if(executionLimits.maxfilesize > 0){ //0 equals no limit
		limit.rlim_cur = executionLimits.maxfilesize;
		limit.rlim_max = executionLimits.maxfilesize;
		setrlimit(RLIMIT_FSIZE, &limit);
	}
	if(executionLimits.maxprocesses > 0){ //0 equals no change
		limit.rlim_cur = executionLimits.maxprocesses;
		limit.rlim_max = executionLimits.maxprocesses;
		setrlimit(RLIMIT_NPROC, &limit);
	}
}

/**
 * Execute program inside jail with proper isolation and error handling
 * This method is called in the child process after fork and never returns
 * @param pm process monitor with prisoner information
 * @param name program to execute
 * @param detail detail string for error messages
 */
void Jail::executeInJail(processMonitor &pm, string name, const char *detail){
	Logger::setForeground(false);
	try {
		// All privileged operations first (require root)
		// Setup cgroup isolation before chroot
		setupCgroup(pm);	
		// Setup namespace isolation to hide unnecessary information
		setupNamespaces();
		 // pivot_root isolation or chroot
		setupFilesystemIsolation(pm);
		// setuid to prisoner (drops root)
		// setupNamespaceUser(pm); STILL NEEDS TESTING
		// become prisoner (drops to prisoner UID/GID)
		pm.becomePrisoner();
		// set resource limits (as prisoner)
		setLimits(pm);
		// exec program (never returns)
		transferExecution(pm, name);
	} catch(const char *s) {
		Logger::log(LOG_ERR, "Error running %s: %s", detail, s);
		printf("\nJail error: %s\n",s);
	} catch(const string &s) {
		Logger::log(LOG_ERR, "Error running %s: %s", detail, s.c_str());
		printf("\nJail error: %s\n", s.c_str());
	} catch(...) {
		Logger::log(LOG_ERR, "Error running %s", detail);
		printf("\nJail error: at execution stage\n");
	}
	_exit(EXIT_SUCCESS);
}

/**
 * run program controlling timeout and redirection
 */
string Jail::run(processMonitor &pm, string name, int othermaxtime, bool VNCLaunch){
	int maxtime;
	pm.getLimits().log("run");
	if (othermaxtime) {
		maxtime = othermaxtime;
		Logger::log(LOG_INFO, "Other maxtime set: %d", othermaxtime);
	}
	else
		maxtime = pm.getMaxTime();
	int fdmaster = -1;
	signal(SIGTERM, SIG_IGN);
	signal(SIGKILL, SIG_IGN);
	newpid = forkpty(&fdmaster, NULL, NULL, NULL);
	if (newpid == -1) { //forkpty error
		Logger::log(LOG_INFO, "Jail: forkpty error %m");
		return "Jail: forkpty error";
	}
	if (newpid == 0) { //new process
		executeInJail(pm, name, (VNCLaunch ? "VNCLaunch" : "run")); // Never returns
	}
	Logger::log(LOG_INFO, "child pid %d",newpid);
	RedirectorTerminalBatch redirector(fdmaster);
	time_t startTime = time(NULL);
	time_t lastTime = startTime;
	int stopSignal = SIGTERM;
	int status;
	while(redirector.isActive()) {
		redirector.advance();
		pid_t wret = waitpid(newpid, &status, WNOHANG);
		if (wret == newpid) {
			if(WIFSIGNALED(status)){
				int signal = WTERMSIG(status);
				char buf[1000];
				sprintf(buf,"\r\nJail: program terminated due to \"%s\" (%d)",
						strsignal(signal),signal);
				redirector.addMessage(buf);
			}else if(WIFEXITED(status)){
				int exitcode = WEXITSTATUS(status);
				if(exitcode != EXIT_SUCCESS){
					char buf[100];
					sprintf(buf,"\r\nJail: program terminated normally with exit code %d.\n",exitcode);
					redirector.addMessage(buf);
				}
			}else{
				redirector.addMessage("\r\nJail: program terminated but unknown reason.");
			}
			newpid = -1;
			break;
		} else if(wret > 0){//waitpid error wret != newpid
			redirector.addMessage("\r\nJail waitpid error: ret>0.\n");
			break;
		} else if(wret == -1) { //waitpid error
			redirector.addMessage("\r\nJail waitpid error: ret==-1.\n");
			Logger::log(LOG_INFO,"Jail waitpid error: %m");
			break;
		}
		if (wret == 0){ //Process running
			time_t now=time(NULL);
			if(lastTime != now){
				int elapsedTime = now - startTime;
				lastTime = now;
				if(elapsedTime > JAIL_MONITORSTART_TIMEOUT && !pm.isMonitored()){
					if(stopSignal != SIGKILL)
						redirector.addMessage("\r\nJail: browser connection error.\n");
					Logger::log(LOG_INFO,"Not monitored");
					kill(newpid, stopSignal);
					stopSignal = SIGKILL; //Second try
				} else if(elapsedTime > maxtime){
					redirector.addMessage("\r\nJail: execution time limit reached.\n");
					Logger::log(LOG_INFO,"Execution time limit (%d) reached"
							,maxtime);
					kill(newpid, stopSignal);
					stopSignal = SIGKILL; //Second try
				} else if(VNCLaunch && redirector.getOutputSize() > 0){
					redirector.advance();
					break;
				}else if(pm.isOutOfMemory()){
					string ml = pm.getMemoryLimit();
					if(stopSignal != SIGKILL)
						redirector.addMessage("\r\nJail: out of memory (" + ml + ")\n");
					Logger::log(LOG_INFO,"Out of memory (%s)", ml.c_str());
					kill(newpid, stopSignal);
					stopSignal = SIGKILL; //Second try
				}
			}
		}
	}
	//wait until 5sg for redirector to read and send program output
	int max_iter = VNCLaunch ? 5 : 50;
	for(int i=0; redirector.isActive() && i < max_iter; i++){
		redirector.advance();
		Util::sleep(100000); // 1/10 sec
	}
	string output = redirector.getOutput();
	Logger::log(LOG_DEBUG,"Complete program output: %s", output.c_str());
	return output;
}

/**
 * run program in terminal controlling timeout and redirection
 */
void Jail::runTerminal(processMonitor &pm, webSocket &ws, string name){
	int fdmaster = -1;
	ExecutionLimits executionLimits = pm.getLimits();
	signal(SIGTERM, SIG_IGN);
	signal(SIGKILL, SIG_IGN);
	newpid = forkpty(&fdmaster, NULL, NULL, NULL);
	if (newpid == -1) { //fork error
		Logger::log(LOG_INFO, "Jail: fork error %m");
		return;
	}
	if (newpid == 0) { //new process
		executeInJail(pm, name, "terminal"); // Never returns
	}
	Logger::log(LOG_INFO, "child pid %d",newpid);
	RedirectorTerminal redirector(fdmaster,&ws);
	Logger::log(LOG_INFO, "Redirector start terminal control");
	time_t startTime = time(NULL);
	time_t lastTime = startTime;
	int stopSignal = SIGTERM;
	int status;
	Logger::log(LOG_INFO,"run: start redirector loop");
	while(redirector.isActive() && !ws.isClosed()){
		redirector.advance();
		pid_t wret = waitpid(newpid, &status, WNOHANG);
		if(wret == 0){
			time_t now = time(NULL);
			if(lastTime != now){
				int elapsedTime = now-startTime;
				lastTime = now;
				if (elapsedTime > JAIL_MONITORSTART_TIMEOUT && !pm.isMonitored()) {
					Logger::log(LOG_INFO, "Not monitored");
					if(stopSignal != SIGKILL)
						redirector.addMessage("\r\nJail: process stopped\n");
					redirector.stop();
					pm.stopPrisonerProcess(stopSignal != SIGKILL);
					kill(newpid, stopSignal);
					stopSignal = SIGKILL;
				} else if(elapsedTime > executionLimits.maxtime) {
					if (stopSignal != SIGKILL)
						redirector.addMessage("\r\nJail: execution time limit reached.\n");
					redirector.stop();
					Logger::log(LOG_INFO, "Execution time limit (%d) reached",
							executionLimits.maxtime);
					pm.stopPrisonerProcess(stopSignal != SIGKILL);
					kill(newpid, stopSignal);
					stopSignal = SIGKILL;
				} else if (pm.isOutOfMemory()) {
					string ml= pm.getMemoryLimit();
					if (stopSignal != SIGKILL)
						redirector.addMessage("\r\nJail: out of memory (" + ml + ")\n");
					Logger::log(LOG_INFO, "Out of memory (%s)",ml.c_str());
					pm.stopPrisonerProcess(stopSignal != SIGKILL);
					kill(newpid, stopSignal);
					stopSignal = SIGKILL; //Second try
				}
			}
		} else { //Not running or error
			break;
		}
	}
	Logger::log(LOG_DEBUG, "End redirector loop");
	//wait until 5sg for redirector to read and send program output
	for (int i = 0; redirector.isActive() && i < 50; i++) {
		redirector.advance();
		Util::sleep(100000); // 1/10 sec
	}
	pm.cleanTask();
}

/**
 * run program in VNCserver controlling timeout and redirection
 */
void Jail::runVNC(processMonitor &pm, webSocket &ws, string name){
	ExecutionLimits executionLimits = pm.getLimits();
	signal(SIGTERM, SIG_IGN);
	signal(SIGKILL, SIG_IGN);
	string output = run(pm, name, 10, true);
	Logger::log(LOG_DEBUG,"VNC launch %s", output.c_str());
	int VNCServerPort = Util::atoi(output);
	RedirectorVNC redirector(&ws, VNCServerPort);
	Logger::log(LOG_INFO, "Redirector start vncserver control");
	time_t startTime=time(NULL);
	time_t lastTime=startTime;
	bool noMonitor=false;
	Logger::log(LOG_INFO,"run: start redirector loop");
	while(redirector.isActive() && !ws.isClosed()){
		redirector.advance();
		time_t now=time(NULL);
		if(lastTime != now){
			int elapsedTime=now-startTime;
			lastTime = now;
			//TODO report to user the out of resources
			if(elapsedTime > executionLimits.maxtime){
				Logger::log(LOG_INFO,"Execution time limit (%d) reached"
						,executionLimits.maxtime);
				redirector.stop();
				break;
			}
			if(pm.isOutOfMemory()){
				string ml= pm.getMemoryLimit();
				Logger::log(LOG_INFO,"Out of memory (%s)", ml.c_str());
				redirector.stop();
				break;
			}
			if(elapsedTime>JAIL_MONITORSTART_TIMEOUT && !pm.isMonitored()){
				Logger::log(LOG_INFO,"Not monitored");
				redirector.stop();
				noMonitor=true;
				break;
			}
		}
	}
	Logger::log(LOG_DEBUG,"End redirector loop");
	//wait until 5sg for redirector to read and send program output
	for(int i=0; redirector.isActive() && i<50; i++){
		redirector.advance();
		Util::sleep(100000); // 1/10 sec
	}
	if(pm.installScript(".vpl_vnc_stopper.sh", "vpl_vnc_stopper.sh")){
		output=run(pm, ".vpl_vnc_stopper.sh", 5); //FIXME use constant
		Logger::log(LOG_DEBUG,"%s",output.c_str());
	}
	if(noMonitor){
		pm.cleanTask();
	}
}

/**
 * Data Passthrough from navigator to local webserver
 */
void Jail::runPassthrough(processMonitor &pm, Socket *s) {
	// TODO change to config data
	string localServerAdress = pm.getLocalWebServer();
	RedirectorWebServer redirector(s, localServerAdress);
	Logger::log(LOG_INFO, "Redirector web server request start");
	time_t startTime = time(NULL);
	time_t lastTime = startTime;
	while (redirector.isActive() && ! s->isClosed()) {
		redirector.advance();
		time_t now = time(NULL);
		if (lastTime != now) {
			int elapsedTime = now - startTime;
			lastTime = now;
			if (elapsedTime > JAIL_HARVEST_TIMEOUT) {
				Logger::log(LOG_INFO,"Execution time limit (%d) reached", JAIL_HARVEST_TIMEOUT);
				s->send("<b>Execution time limit reached</b>");
				redirector.stop();
				break;
			}
		}
	}
	Logger::log(LOG_DEBUG,"End web redirector loop");
}
